/* Copyright (c) 2017-2018 VMware, Inc. All rights reserved. -- VMware Confidential */
import * as ts from "typescript";

import {NewTransInfo, TransformationInfo} from "../transformationInfo";

interface TreeNode {
   methodName: string,
   args: string[],
   subStr: string,
   position: number,
   line: number,
   character: number
}

export interface SubContentTransformation {
   oldContent: string,
   newContent: string,
   position: number,
   errorMsg: string,
   line: number,
   character: number
}

export class FileMigrator {
   filePath: string;
   fileContent: string;

   constructor(filePath: string, fileContent: string) {
      this.filePath = filePath;
      this.fileContent = fileContent;
   }

   getContent(): string {
      return this.fileContent;
   }

   setContent(content: string) {
      this.fileContent = content;
   }

   getPath(): string {
      return this.filePath;
   }

   migrateFile(transformationInfo: TransformationInfo): void {
      const ast: ts.SourceFile = this.createAST();

      this.applyTransformations(this.getAllTransformations(ast, transformationInfo));
   }

   // this method should be extended with each another use case of finding our function calls;
   private getAllTransformations(ast: ts.SourceFile, transformationInfo: TransformationInfo): SubContentTransformation[] {
      const nodes: TreeNode[] = this.getAllFunctionCallsByObject(ast, transformationInfo.getObjects());
      const errors = transformationInfo.getErrorMessages();
      const contentTransfromations: SubContentTransformation[] = nodes.map((node) => {
         const newContent = FileMigrator.constructFinalSubStr(node, transformationInfo);
         let errorMsg = null;
         if (newContent === "") {
            errorMsg = (errors.hasOwnProperty(node.methodName)) ? errors[node.methodName]
               : `There is no valid transformation for '${node.subStr}'`;
         }
         const obj = {
            oldContent: node.subStr,
            newContent: newContent,
            position: node.position,
            errorMsg: errorMsg,
            line: node.line,
            character: node.character
         };
         return obj;
      });

      return contentTransfromations;
   }

   getAllFileTransformations(transformationInfo: TransformationInfo): SubContentTransformation[] {
      const ast: ts.SourceFile = this.createAST();

      const contentTransformations: SubContentTransformation[] = this.getAllTransformations(ast, transformationInfo);

      return contentTransformations;
   }

   applyTransformations(subContentTransformations: SubContentTransformation[]): void {
      this.sortByPosition(subContentTransformations).forEach((contentT) => {
         if (!contentT.errorMsg) {
            // the replacement is done as follows: Since the API needed to be replaced
            // is starting from the position contentT.position we are going to get the whole content
            // up to position contentT.position and we are going to change in the content from
            // the position point to the end the API function
            const prefixContent: string = this.getContent().substring(0, contentT.position);
            const suffixContent: string = this.getContent().substring(contentT.position);
            this.fileContent = prefixContent +
               suffixContent.replace(contentT.oldContent, contentT.newContent);
         } else {
            console.error(contentT.errorMsg);
         }
      });
   }

   private createAST(): ts.SourceFile {
      const sourceFile = ts.createSourceFile(this.getPath(), this.getContent(),
         ts.ScriptTarget.ES2015, /*setParentNodes */ true);
      return sourceFile;
   }

   private getAllFunctionCallsByObject(sourceFile: ts.SourceFile, objects: string[]): TreeNode[] {
      const FUNCTION_ARGUMENTS = 2;
      let results: TreeNode[] = [];
      let self = this;
      delintNode(sourceFile);
      return results;

      function delintNode(node: ts.Node) {
         if (!containsObjectText(node)) {
            return;
         }
         if (isFunctionCallFromObject(node)) {
            const propertyAccessExpression: ts.Node = node.getChildAt(0);
            if (endsWithObjectText(propertyAccessExpression.getChildAt(0))) {
               const methodName: string = parseMethodFromPropertyExpression(propertyAccessExpression);
               results.push({
                  methodName: methodName,
                  args: parseArgsFromCallExpression(node),
                  subStr: self.getContent().substr(node.getStart(), node.getWidth()),
                  position: node.getStart(),
                  line: sourceFile.getLineAndCharacterOfPosition(node.getStart()).line,
                  character: sourceFile.getLineAndCharacterOfPosition(node.getStart()).character
               });
               return;
            }
         }

         for (const n of node.getChildren()) {
            delintNode(n);
         }
      }

      function containsObjectText(node: ts.Node): boolean {
         return (
            objects.filter((object) => {
               return (node.getText().indexOf(object) >= 0);
            }).length > 0);
      }

      function isFunctionCallFromObject(node: ts.Node): boolean {
         return node.kind === ts.SyntaxKind.CallExpression &&
            node.getChildAt(0).kind === ts.SyntaxKind.PropertyAccessExpression;
      }

      function endsWithObjectText(node: ts.Node): boolean {
         return (
            objects.filter((object) => {
               return node.getFullText().endsWith(object);
            }).length > 0);
      }

      function parseMethodFromPropertyExpression(node: ts.Node): string {
         const METHOD_CHILD_IN_PROPERTY_EXPRESSION: number = 2;
         return node.getChildAt(METHOD_CHILD_IN_PROPERTY_EXPRESSION).getText();
      }

      function parseArgsFromCallExpression(node: ts.Node): string[] {
         const argsNode: ts.Node = node.getChildAt(FUNCTION_ARGUMENTS);
         const children = argsNode.getChildren()
            .filter((node) => {
               return node.kind !== ts.SyntaxKind.CommaToken;
            });
         for (node of children) {
            delintNode(node);
         }
         return children.map<string>(
            (item) => {
               return item.getText();
            });
      }
   }

   private sortByPosition(contentTransfromations: SubContentTransformation[]): SubContentTransformation[] {
      const sorted: SubContentTransformation[] =
         contentTransfromations.slice()
            .sort((a, b) => {
               return (a.position < b.position) ? -1 : 1;
            });
      return sorted;
   }

   private static constructFinalSubStr(node: TreeNode, transform: TransformationInfo): string {
      const newTransInfo: NewTransInfo = transform.getFunctionTransfromations()[node.methodName];
      if (!newTransInfo) {
         return "";
      }
      let replacedNewArgs: string = newTransInfo.newArgs;
      // there are APIs which have optional last argument.
      if (newTransInfo.oldArgs && (newTransInfo.oldArgs.length !== node.args.length)) {
         node.args.push(null);
      }
      for (let i = 0; i < node.args.length; ++i) {
         replacedNewArgs = replacedNewArgs.replace("$" + newTransInfo.oldArgs[i], node.args[i]);
      }
      const finalString: string = transform.getNamespace() + "." + newTransInfo.newName + "(" + (replacedNewArgs || "") + ")" + (newTransInfo.postfix || "");

      return finalString;
   }

   static countNewLines(content: string): number {
      return content.split(/\r\n|\r|\n/).length;
   }

   static countCharactersFromLastNewLine(content: string): number {
      const lastIndex = content.lastIndexOf("\n") + 1;
      const character = content.substring(lastIndex).length;
      return character;
   }
}